/* This file contains the definitions for the BruteForceSudokuSolver class, which solves
 * a Sudoku puzzle using a brute force algorithm.  The algorithm came from some example
 * code in C on Wikipedia:
 * http://en.wikipedia.org/wiki/Algorithmics_of_sudoku
 *
 * Date       User              Note
 * 2009-11-04 Eric Oulashin     Created
 * 2009-11-05 Eric Oulashin     Added mNumSolutions and mSolutions.  Updated solve() to copy the mPuzzle values to
 *                              mSolvedPuzzle.
 * 2009-11-06 Eric Oulashin     Added reset()
 */

#include "BruteForceSudokuSolver.h"
#include <cstdlib>

BruteForceSudokuSolver::BruteForceSudokuSolver()
	: mNumSolutions(0),
	  mSolutions(NULL)
{
	for (int row = 0; row < 9; ++row)
	{
		for (int col = 0; col < 9; ++col)
			mPuzzle[row][col] = 0;
	}
}

BruteForceSudokuSolver::~BruteForceSudokuSolver()
{
	delete [] mSolutions;
}

void BruteForceSudokuSolver::setVal(int pRow, int pCol, int pVal)
{
	if ((pVal < 1) || (pVal > 9))
		return;

	if ((pRow >= 0) && (pRow < 9) && (pCol >= 0) && (pCol < 9))
		mPuzzle[pRow][pCol] = pVal;
}

int BruteForceSudokuSolver::getVal(int pRow, int pCol) const
{
	if ((pRow >= 0) && (pRow < 9) && (pCol >= 0) && (pCol < 9))
		return (mPuzzle[pRow][pCol]);
	else
		return -1;
}

int BruteForceSudokuSolver::NumSolutions() const
{
	return mNumSolutions;
}

int BruteForceSudokuSolver::getSolvedVal(int pPuzzleIndex, int pRow, int pCol) const
{
	if ((pPuzzleIndex >= 0) && (pPuzzleIndex < mNumSolutions) && (mNumSolutions > 0) && (pRow >= 0) && (pRow < 9) && (pCol >= 0) && (pCol < 9))
		return (mSolutions[pPuzzleIndex].GetValue(pRow, pCol));
	else
		return -1;
}

void BruteForceSudokuSolver::solve()
{
	solve(0, 0);
}

void BruteForceSudokuSolver::reset()
{
	for (int row = 0; row < 9; ++row)
	{
		for (int col = 0; col < 9; ++col)
			mPuzzle[row][col] = 0;
	}
	mNumSolutions = 0;
	delete [] mSolutions;
	mSolutions = NULL;
}

int BruteForceSudokuSolver::numGivenNumbers() const
{
	int numNumbersSet = 0;

	for (int row = 0; row < 9; ++row)
	{
		for (int col = 0; col < 9; ++col)
		{
			if ((mPuzzle[row][col] > 0) && (mPuzzle[row][col] < 10))
				++numNumbersSet;
		}
	}

	return numNumbersSet;
}

///////////////////////
// Private functions //
///////////////////////

void BruteForceSudokuSolver::solve(int pRow, int pCol)
{
	if (pRow == 9)
	{
		SudokuSolution *newSolutions = new SudokuSolution[mNumSolutions + 1];
		if (mSolutions != NULL)
		{
			if (mNumSolutions > 0)
			{
				for (int i = 0; i < mNumSolutions; ++i)
					newSolutions[i] = mSolutions[i];
			}
			delete [] mSolutions;
		}
		mSolutions = newSolutions;
		int lastSolutionIndex = mNumSolutions;
		++mNumSolutions;
		// Copy the values from mPuzzle to the last solution in mSolutions.
		for (int row = 0; row < 9; ++row)
		{
			for (int col = 0; col < 9; ++col)
				mSolutions[lastSolutionIndex].SetValue(row, col, mPuzzle[row][col]);
		}
	}
	else
	{
		int t = 0;
		for (int value = 1; value <= 9; ++value)
		{
			if (valSafe(pRow, pCol, value))
			{
				t = mPuzzle[pRow][pCol];
				mPuzzle[pRow][pCol] = value;
				if (pCol == 8)
					solve(pRow + 1, 0);
				else
					solve(pRow, pCol + 1);

				mPuzzle[pRow][pCol] = t;
			}
		}
	}
}

bool BruteForceSudokuSolver::valSafe(int pRow, int pCol, int pVal) const
{
	int row = 0, col = 0;

	if (mPuzzle[pRow][pCol] == pVal) return true;
	if (mPuzzle[pRow][pCol] != 0) return false;
	for (col = 0; col < 9; ++col)
		if (mPuzzle[pRow][col] == pVal) return false;
	for (row = 0; row < 9; ++row)
		if (mPuzzle[row][pCol] == pVal) return false;
	int br = pRow / 3;
	int bc = pCol / 3;
	for (row = br * 3; row < (br + 1) * 3; ++row)
	{
		for (col = bc * 3; col < (bc + 1) * 3; ++col)
			if (mPuzzle[row][col] == pVal) return false;
	}

	return true;
}